unit cloThdU;

// Copyright  2001 by Ziff-Davis, Inc.
// Written by Neil J. Rubenking

interface
uses Windows, SysUtils, Classes;

type
  // This thread type attempts to close a running program. It waits
  // a specified number of seconds, and avoids hanging due to
  // attempting communication with a hung program
  TCloThread = class(TThread)
  private
    fHan  : HWnd;          // handle of window to close
    fPID  : DWORD;         // Process ID of window's process
    fRslt : DWORD;         // Calling program can check result
    fSecs : DWORD;         // Desired seconds to wait
    fNote : TNotifyEvent;  // Callback for seconds countdown
    fExp  : Boolean;       // true if is an Explorer window
    procedure SyncNote;
  public
    constructor Create(vHan : HWnd; vPID, vSecs : DWORD; vRet,
      vNote : TNotifyEvent; vExp : Boolean);
    procedure Execute; override;
    property Rslt : DWORD read fRslt;
    property Seconds : DWORD read fSecs;
  end;
VAR
  StopEv : THandle;

implementation
uses Messages;

{ TCloThread }

type
  // This worker thread actually performs the close operation
  TSMTOThread = class(TThread)
  private
    fHan  : HWnd;
    fTime : DWORD;
    fExp  : Boolean;
  public
    constructor Create(vHan : HWnd; vTime : DWORD; vExp : Boolean);
    procedure Execute; override;
  end;


constructor TCloThread.Create(vHan : HWnd; vPID, vSecs : DWORD;
  vRet, vNote : TNotifyEvent; vExp : Boolean);
begin
  Inherited Create(False);
  FreeOnTerminate := True;
  fHan            := vHan;
  fPid            := vPid;
  fSecs           := vSecs;
  OnTerminate     := vRet;
  fNote           := vNote;
  fExp            := vExp;
end;

procedure TCloThread.SyncNote;
begin
  IF Assigned(fNote) THEN
    fNote(Self);
end;

procedure TCloThread.Execute;
VAR fobjs : ARRAY[0..2] OF Integer;
begin
  fObjs[0] := OpenProcess(Windows.SYNCHRONIZE, False, fPID);
  fObjs[1] := StopEv;
  fObjs[2] := TSMTOThread.Create(fHan, 1000*fSecs, fExp).Handle;
  Synchronize(SyncNote);
  REPEAT
    fRslt := WaitForMultipleObjects(3, @fObjs, false, 1000);
    IF fRslt <> WAIT_TIMEOUT THEN Exit;
    Dec(fSecs);
    Synchronize(SyncNote);
  UNTIL fSecs = 0;
end;

{ TSMTOThread }

constructor TSMTOThread.Create(vHan: HWnd; vTime : DWORD;
  vExp : Boolean);
begin
  Inherited Create(False);
  FreeOnTerminate := True;
  fHan            := vHan;
  fTime           := vTime;
  fExp            := vExp;
end;

procedure TSMTOThread.Execute;
VAR D : DWord;
begin
  SetLastError(0);
  // Try an innocuous message - if it times out the program is hung
  SendMessageTimeout(fHan, WM_GETTEXTLENGTH, 0, 0,
    SMTO_ABORTIFHUNG, 100, D);
  // If the window seems OK, bring it to the top
  IF GetLastError = 0 THEN
    IF SetForegroundWindow(fHan) THEN
      BringWindowToTop(fHan);
  SetLastError(0);
  IF fExp THEN
    begin
      // Explorer windows need a little extra to close 'em
      SendMessageTimeout(fHan, WM_DESTROY, 0, 0, SMTO_ABORTIFHUNG,
        100, D);
      Sleep(100);
      SendMessageTimeout(fHan, WM_NCDESTROY, 0, 0, SMTO_ABORTIFHUNG,
        100, D);
      Sleep(100);
    end;
  // Now send the WM_CLOSE message with timeout
  SendMessageTimeOut(fHan, WM_CLOSE, 0, 0, SMTO_ABORTIFHUNG,
    fTime, D);
end;

initialization
  StopEv := CreateEvent(nil, true, false, 'CloseThread Stop Event '+
    '{3F1B2C4E-2847-4EE6-AE96-7AB454D22EB4}');
finalization
  CloseHandle(StopEv);
end.
